// -----------------------------------------------------------------------
// Author: Jianxin wu (wujx2001@gmail.com)
// This file is provided to simplify the usage of the provided
// "Visual Place Categorization" dataset.
// For academic and research purpose only.
// -----------------------------------------------------------------------
// Please read carefully the comments to variable and functions for usage.
// -----------------------------------------------------------------------

#ifndef __VPC_H_
#define __VPC_H_

#include <string>
#include <vector>

// total number of semantic category (room type) in our dataset
const int VPC_numCategory = 12;
// total number of homes in which we have captured videos
const int VPC_numHomes = 6;
// names of the semantic categories
// "transition" is a special category: for frames which are hard to categorize, 
// e.g. frames contain visual contents from two room categories, or,
// room that are not typical (hard to associate with any category)
const std::string VPC_Categories[VPC_numCategory] = {
        "transition", // make sure that "transition" is the first one (index is 0)
        "bedroom",
        "bathroom",
        "kitchen",
        "closet",
        "living_room",
        "dining_room",
        "family_room",
        "workspace",
        "exercise_room",
        "media_room",
        "corridor"
};
// Data structure that summarizes category labels of all frames inside a home
// A home may have several floors: 1st floor, 2nd floor, and basement
// All frames from a home is stored in a directory 'Home?/', where ?=1..VPC_numHomes
// 1st floor frames are in 'Home?/1/', 2nd floor in 'Home?/2/, and basement in 'Home?/0/'
// 'num_dirs' is the number of sub-directories (aka floors) in a home
// 'dirs' and 'labels' both have 'num_dirs' elements
// 'dirs' store the directory name of sub-directories, e.g. dirs[0]="1/"
//  --- NOTE: there is a offset because a C array starting from index 0
// 'labels' store category labels, e.g. if "1/" contains x frames
// then labels[0] is a vector of length x, and labels[0][x] is label of x-th frame in "1/"
struct VPC_Input
{
    int num_dirs;
    std::vector<std::string> dirs; // sub-directory names
    std::vector< std::vector<int> > labels; // label of images from each sub-directory
    // constructor, destructor, and help function to clear the contents
    VPC_Input():num_dirs(0)
    {
    }
    ~VPC_Input()
    {
        Clear();
    }
    void Clear();
};

// see VPC_IO.cpp for details documentation of functions

// Mapping from a semantic category name string to an index number.
int VPC_CategoryToIndex(const std::string category_name);
// Read the groundtruth labels into 'input. Labels are in 'label.txt' in each directory.
int VPC_ReadLabelFile(const int home,VPC_Input& input);
// Get full pathname of a frame using: input (representing a home), dir (floor index), & image (frame index inside a floor)
int VPC_GetFileName(VPC_Input& input, const int dir,const int image,std::string& filename);
// Compute the number of frames that come from a home that is in 'homes'
// & whose semantic category is in "category"
int VPC_ValidCount(std::vector<bool>& homes,std::vector<bool>& category);

// Providing a way to sequentially traverse through specified homes and room types
// so that you can run your method on the VPC dataset
int VPC_Traverse(std::vector<bool>& homes,std::vector<bool>& category);

// This function provides a suggested way to evaluate your VPC method when frames are considered independently.
void VPC_Evaludate_Single_Frame(const char* outputfile);

#endif // __VPC_H_
